'use client';

import React, { useState, useEffect, useRef } from 'react';
import { io, Socket } from 'socket.io-client';
import { useParams } from 'next/navigation';
import {
  User,
  Plane,
  Calendar,
  Clock,
  MapPin,
  FileText,
  Download,
  RotateCcw,
  CheckCircle,
  XCircle,
  Smartphone,
  Armchair,
  AlertTriangle,
  Settings,
  Loader,
  Phone,
  Eye,
  Activity,
  Wifi,
  WifiOff,
  Archive,
  Play,
  Square,
  Pause
} from 'lucide-react';
import DetailLayout from '@/components/ui/DetailLayout';
import { renderCustomer, renderStatusBadge, checkinStatusConfig } from '@/components/ui/TableRenderers';
import type { CheckinTicket } from '@/types';
import PermissionGuard from '@/components/guards/PermissionGuard';

// Havayolu temları
const airlineThemes = {
  THY: {
    primary: 'bg-red-600',
    secondary: 'bg-red-100',
    text: 'text-red-600',
    border: 'border-red-200',
    accent: 'bg-red-50',
    name: 'Turkish Airlines',
    logo: '🇹🇷'
  },
  PC: {
    primary: 'bg-yellow-500',
    secondary: 'bg-yellow-100',
    text: 'text-yellow-600',
    border: 'border-yellow-200',
    accent: 'bg-yellow-50',
    name: 'Pegasus Airlines',
    logo: '🐎'
  },
  AJET: {
    primary: 'bg-orange-600',
    secondary: 'bg-orange-100',
    text: 'text-orange-600',
    border: 'border-orange-200',
    accent: 'bg-orange-50',
    name: 'AnadoluJet',
    logo: '🔶'
  }
};

interface Comment {
  id: string;
  author: string;
  content: string;
  timestamp: Date;
  type: 'comment' | 'system' | 'status_change';
  isInternal?: boolean;
}

interface CheckinProgress {
  step: number;
  title: string;
  description: string;
  status: 'pending' | 'processing' | 'completed' | 'failed';
  timestamp?: Date;
  details?: string;
  error?: string;
  screenshot?: string;
}

interface CheckinLog {
  id: string;
  timestamp: Date;
  level: 'info' | 'warn' | 'error' | 'debug';
  action: string;
  message: string;
  details?: string;
  screenshot?: string;
}

type ConnectionStatus = 'connected' | 'disconnected' | 'connecting';

export default function CheckinDetailPage() {
  return (
    <PermissionGuard requiredPermissions={['checkin_view', 'checkin_view_own', 'checkin_view_all']}>
      <CheckinDetailPageContent />
    </PermissionGuard>
  );
}

function CheckinDetailPageContent() {
  const params = useParams();
  const checkinId = params.id as string;

  const [checkin, setCheckin] = useState<CheckinTicket | null>(null);
  const [loading, setLoading] = useState(true);
  const [comments, setComments] = useState<Comment[]>([]);
  const [progress, setProgress] = useState<CheckinProgress[]>([]);
  const [logs, setLogs] = useState<CheckinLog[]>([]);
  const [currentJobId, setCurrentJobId] = useState<string | null>(null);
  const [showLogs, setShowLogs] = useState(false);
  const [connectionStatus, setConnectionStatus] = useState<ConnectionStatus>('disconnected');

  // Havayolu temasını al
  const getAirlineTheme = (airline: string) => {
    return airlineThemes[airline as keyof typeof airlineThemes] || airlineThemes.THY;
  };
  const [isRetrying, setIsRetrying] = useState(false);
  const socketRef = useRef<Socket | null>(null);

  useEffect(() => {
    fetchCheckinDetails();
    connectWebSocket();
    fetchLogsFromDatabase(); // DB'den logları yükle

    return () => {
      if (socketRef.current) {
        socketRef.current.disconnect();
      }
    };
  }, [checkinId]);

  // Check-in verisi yüklendikten sonra yorumları ve progress'i başlat
  useEffect(() => {
    if (checkin) {
      fetchComments();
      initializeProgress();
    }
  }, [checkin]);

  const connectWebSocket = () => {
    if (socketRef.current?.connected) return;

    setConnectionStatus('connecting');
    const socket = io(process.env.NEXT_PUBLIC_API_URL || 'http://localhost:3002'); // Backend port'u düzelt

    socket.on('connect', () => {
      setConnectionStatus('connected');
      console.log('Socket.IO connected');
    });

    socket.on('checkin_status', (data) => {
      try {
        // Sadece bu check-in'e ait mesajları işle
        if (data.jobId && currentJobId && currentJobId !== data.jobId) {
          return;
        }
        console.log('📡 Progress update received:', data);
        handleWebSocketMessage({ type: 'progress_update', ...data });
      } catch (error) {
        console.error('Socket message parse error:', error);
      }
    });

    socket.on('checkin_result', (data) => {
      try {
        // Sadece bu check-in'e ait mesajları işle
        if (data.jobId && currentJobId && currentJobId !== data.jobId) {
          return;
        }
        console.log('🏁 Checkin result received:', data);
        handleWebSocketMessage({ type: 'checkin_complete', ...data });
      } catch (error) {
        console.error('Socket message parse error:', error);
      }
    });

    socket.on('disconnect', () => {
      setConnectionStatus('disconnected');
      // Socket.IO handles reconnection automatically
    });

    socket.on('connect_error', (error) => {
      console.error('Socket.IO connection error:', error);
      setConnectionStatus('disconnected');
    });

    socketRef.current = socket;
  };

  const addProgress = (percentage: number, message: string, status: string = 'processing') => {
    const now = new Date();

    // Yüzdeye göre hangi ana adımda olduğumuzu belirle
    let mainStep = 1;
    if (percentage >= 95) mainStep = 5; // Bildirim Gönderimi
    else if (percentage >= 85) mainStep = 4; // Koltuk & Biniş Kartı
    else if (percentage >= 75) mainStep = 3; // Check-in İşlemi
    else if (percentage >= 70) mainStep = 2; // Rezervasyon Sorgulama
    else mainStep = 1; // Bağlantı

    // Ana adımı güncelle
    setProgress(prev => prev.map(p => {
      if (p.step === mainStep) {
        return {
          ...p,
          status: percentage === 100 ? 'completed' : 'processing',
          timestamp: now,
          details: message
        };
      }
      // Önceki adımları tamamlandı olarak işaretle
      if (p.step < mainStep) {
        return { ...p, status: 'completed' as const };
      }
      return p;
    }));

    // Detaylı log ekle
    const newLog: CheckinLog = {
      id: Date.now().toString(),
      timestamp: now,
      level: status === 'failed' ? 'error' : status === 'completed' ? 'info' : 'info',
      action: `${percentage}%`,
      message: message,
      details: message
    };
    setLogs(prev => [newLog, ...prev].slice(0, 100));

    console.log('📊 Progress & Log updated:', { step: mainStep, percentage, message });
  };

  const handleWebSocketMessage = (data: {
    type: string;
    message?: string;
    progress?: number;
    status?: string;
    log?: { timestamp: string; message: string; level: string };
    passengerId?: string;
    result?: unknown;
    success?: boolean;
  }) => {
    switch (data.type) {
      case 'progress_update':
        // Backend'den gelen message ve progress'i progress sistemine entegre et
        if (data.message && data.progress !== undefined) {
          addProgress(data.progress, data.message, data.status || 'processing');
        }
        break;
      case 'status_change':
        if (checkin && data.status) {
          setCheckin(prev => prev ? { ...prev, checkinStatus: data.status as CheckinTicket['checkinStatus'], updatedAt: new Date() } : null);
        }
        break;
      case 'log_entry':
        if (data.log) {
          addLog(data.log);
        }
        break;
      case 'passenger_result':
        updatePassengerResult(data.passengerId, data.result);
        break;
      case 'checkin_complete':
        // Check-in tamamlandığında status'u güncelle
        console.log('🎉 Check-in completed:', data);
        if (checkin) {
          const finalStatus = data.success ? 'completed' : 'failed';
          setCheckin(prev => prev ? {
            ...prev,
            checkinStatus: finalStatus,
            updatedAt: new Date(),
            boardingPassFile: data.flightInfo?.boardingPassFile,
            smsStatus: data.flightInfo?.smsStatus
          } : null);

          // Final progress update
          addProgress(100, data.success ? '✅ Check-in başarıyla tamamlandı!' : '❌ Check-in başarısız', finalStatus);

          // Check-in sonucu için comment ekle
          const resultComment: Comment = {
            id: `result_${Date.now()}`,
            author: 'Sistem',
            content: data.success
              ? `✅ Check-in başarıyla tamamlandı! ${data.message || ''}`
              : `❌ Check-in başarısız: ${data.message || 'Bilinmeyen hata'}`,
            timestamp: new Date(),
            type: 'system'
          };
          setComments(prev => [...prev, resultComment]);

          // LocalStorage'ı güncelle
          try {
            const storedData = localStorage.getItem('checkin-tickets');
            if (storedData) {
              const tickets = JSON.parse(storedData);
              const updatedTickets = tickets.map((ticket: { id: string; checkinStatus: string; updatedAt: string }) =>
                ticket.id === checkinId
                  ? { ...ticket, checkinStatus: finalStatus, updatedAt: new Date().toISOString() }
                  : ticket
              );
              localStorage.setItem('checkin-tickets', JSON.stringify(updatedTickets));
            }
          } catch (error) {
            console.error('LocalStorage update error:', error);
          }
        }
        break;
      default:
        console.log('Unknown WebSocket message type:', data.type);
    }
  };

  const initializeProgress = () => {
    const airline = checkin?.airline || 'THY';

    const initialProgress: CheckinProgress[] = [
      {
        step: 1,
        title: `${airline} Bağlantısı`,
        description: airline === 'THY' ? 'Turkish Airlines Hero Check-in başlatılıyor' :
                    airline === 'PC' ? 'Pegasus Airlines check-in başlatılıyor' :
                    'Havayolu check-in başlatılıyor',
        status: 'pending'
      },
      {
        step: 2,
        title: 'Rezervasyon Sorgulama',
        description: 'PNR ve yolcu bilgileri ile rezervasyon sorgulanıyor',
        status: 'pending'
      },
      {
        step: 3,
        title: 'Check-in İşlemi',
        description: 'Otomatik check-in işlemi gerçekleştiriliyor',
        status: 'pending'
      },
      {
        step: 4,
        title: 'Koltuk & Biniş Kartı',
        description: 'Koltuk seçimi ve biniş kartları hazırlanıyor',
        status: 'pending'
      },
      {
        step: 5,
        title: 'Bildirim Gönderimi',
        description: 'Müşteriye SMS/e-posta bildirimi gönderiliyor',
        status: 'pending'
      }
    ];
    setProgress(initialProgress);
  };

  const updateProgress = (step: number, status: CheckinProgress['status'], details?: string, error?: string) => {
    setProgress(prev => prev.map(p =>
      p.step === step
        ? { ...p, status, details, error, timestamp: new Date() }
        : p
    ));
  };

  const addLog = (log: CheckinLog) => {
    setLogs(prev => [log, ...prev].slice(0, 100)); // Keep last 100 logs
  };

  const updatePassengerResult = (passengerId: string, result: Record<string, unknown>) => {
    if (!checkin) return;

    setCheckin(prev => prev ? {
      ...prev,
      passengerResults: prev.passengerResults.map(pr =>
        pr.passengerId === passengerId ? { ...pr, ...result } : pr
      )
    } : null);
  };

  const fetchLogsFromDatabase = async () => {
    // currentJobId varsa DB'den logları çek
    if (!currentJobId) return;

    try {
      const response = await fetch(`${process.env.NEXT_PUBLIC_API_URL || 'http://localhost:3002'}/api/checkin/logs/${currentJobId}`);
      if (response.ok) {
        const data = await response.json();
        if (data.success && data.logs) {
          // DB loglarını frontend formatına çevir
          const dbLogs = data.logs.map((log: {
            id: number;
            timestamp: string;
            level: string;
            percentage: number;
            message: string;
            details?: string;
          }) => ({
            id: log.id.toString(),
            timestamp: new Date(log.timestamp),
            level: log.level as 'info' | 'warn' | 'error' | 'debug',
            action: `${log.percentage}%`,
            message: log.message,
            details: log.details
          }));
          setLogs(dbLogs);
          console.log('📊 DB logları yüklendi:', dbLogs.length);
        }
      }
    } catch (error) {
      console.error('❌ DB log yükleme hatası:', error);
    }
  };

  const fetchCheckinDetails = async () => {
    try {
      setLoading(true);

      // localStorage'den gerçek veriyi çek
      const savedTickets = localStorage.getItem('checkin-tickets');
      if (savedTickets) {
        try {
          const parsedTickets = JSON.parse(savedTickets).map((ticket: {
            departureDate: string;
            checkinOpenTime: string;
            createdAt: string;
            updatedAt: string;
          }) => ({
            ...ticket,
            departureDate: new Date(ticket.departureDate),
            checkinOpenTime: new Date(ticket.checkinOpenTime),
            createdAt: new Date(ticket.createdAt),
            updatedAt: new Date(ticket.updatedAt)
          }));

          const foundCheckin = parsedTickets.find((ticket: CheckinTicket) => ticket.id === checkinId);
          if (foundCheckin) {
            setCheckin(foundCheckin);
          } else {
            setCheckin(null); // Veri bulunamadı
          }
        } catch (error) {
          console.error('LocalStorage veri parse hatası:', error);
          setCheckin(null);
        }
      } else {
        setCheckin(null); // localStorage'de veri yok
      }
    } catch (error) {
      console.error('Check-in detay çekme hatası:', error);
      setCheckin(null);
    } finally {
      setLoading(false);
    }
  };

  const fetchComments = async () => {
    // Gerçek check-in verilerine göre dinamik yorumlar
    if (!checkin) {
      setComments([]);
      return;
    }

    const dynamicComments: Comment[] = [];

    // Check-in oluşturulma yorumu
    dynamicComments.push({
      id: 'create',
      author: 'Sistem',
      content: `Check-in kaydı oluşturuldu - ${checkin.airline} ${checkin.flightNumber}`,
      timestamp: checkin.createdAt,
      type: 'system'
    });

    // Otomatik check-in etkinse yorum ekle
    if (checkin.isAutoCheckinEnabled) {
      dynamicComments.push({
        id: 'auto_enabled',
        author: 'Sistem',
        content: 'Otomatik check-in etkinleştirildi',
        timestamp: checkin.createdAt,
        type: 'system'
      });
    }

    // Check-in tamamlandıysa yorum ekle
    if (checkin.checkinStatus === 'completed') {
      dynamicComments.push({
        id: 'completed',
        author: 'Sistem',
        content: `Check-in başarıyla tamamlandı - ${checkin.passengers.length} yolcu`,
        timestamp: checkin.updatedAt,
        type: 'status_change'
      });

      // SMS gönderildiyse yorum ekle
      if (checkin.smsStatus === 'sent') {
        dynamicComments.push({
          id: 'sms_sent',
          author: 'Sistem',
          content: 'SMS bildirimi gönderildi',
          timestamp: checkin.updatedAt,
          type: 'system'
        });
      }
    }

    // Check-in başarısızsa hata yorumu ekle
    if (checkin.checkinStatus === 'failed' && checkin.errorMessage) {
      dynamicComments.push({
        id: 'failed',
        author: 'Sistem',
        content: `Check-in başarısız: ${checkin.errorMessage}`,
        timestamp: checkin.updatedAt,
        type: 'system'
      });
    }

    setComments(dynamicComments);
  };

  const handleAddComment = (content: string, isInternal: boolean) => {
    const newComment: Comment = {
      id: Date.now().toString(),
      author: 'Agent Mehmet',
      content,
      timestamp: new Date(),
      type: 'comment',
      isInternal
    };
    setComments(prev => [...prev, newComment]);
  };

  const handleStatusChange = (newStatus: string) => {
    if (!checkin) return;
    setCheckin(prev => prev ? { ...prev, checkinStatus: newStatus as CheckinTicket['checkinStatus'], updatedAt: new Date() } : null);
  };

  const handleRetryCheckin = async () => {
    if (!checkin || isRetrying) return;

    setIsRetrying(true);

    try {
      setCheckin(prev => prev ? {
        ...prev,
        checkinStatus: 'processing',
        checkinAttempts: prev.checkinAttempts + 1,
        updatedAt: new Date()
      } : null);

      // Reset progress
      initializeProgress();

      // Add system comment
      const retryComment: Comment = {
        id: Date.now().toString(),
        author: 'Sistem',
        content: `Check-in yeniden deneniyor (${checkin.checkinAttempts + 1}. deneme)`,
        timestamp: new Date(),
        type: 'system'
      };
      setComments(prev => [...prev, retryComment]);

      // Simulate API call to restart check-in process
      await fetch(`/api/checkin/${checkinId}/retry`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' }
      });

    } catch (error) {
      console.error('Retry failed:', error);
    } finally {
      setIsRetrying(false);
    }
  };

  const handleDownloadBoardingPass = (passengerId: string) => {
    const result = checkin?.passengerResults.find(r => r.passengerId === passengerId);
    if (result?.boardingPassUrl) {
      window.open(result.boardingPassUrl, '_blank');
    }
  };

  // Check-in kontrol fonksiyonları
  const handleStartCheckin = async () => {
    if (!checkin) return;

    try {
      console.log(`🚀 Check-in başlatılıyor: ${checkin.pnr}`);

      // Durumu processing'e güncelle
      setCheckin(prev => prev ? {
        ...prev,
        checkinStatus: 'processing' as const,
        checkinAttempts: prev.checkinAttempts + 1,
        updatedAt: new Date()
      } : null);

      // Progress'i sıfırla
      initializeProgress();

      // API çağrısı
      const response = await fetch(`${process.env.NEXT_PUBLIC_API_URL || 'http://localhost:3002'}/api/checkin`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          pnr: checkin.pnr,
          lastName: checkin.passengers[0]?.lastName || '',
          phone: checkin.passengers[0]?.phone || '',
          airline: checkin.airline
        }),
      });

      if (response.ok) {
        const result = await response.json();
        console.log('Check-in başlatıldı:', result);

        // JobId'yi kaydet WebSocket filtrelemesi için
        if (result.jobId) {
          setCurrentJobId(result.jobId);
          console.log('💾 JobId saved for progress tracking:', result.jobId);
        }
      } else {
        throw new Error('Check-in başlatılamadı');
      }

    } catch (error) {
      console.error(`❌ Check-in başlatma hatası: ${checkin.pnr}`, error);

      // Hata durumunda failed olarak işaretle
      setCheckin(prev => prev ? {
        ...prev,
        checkinStatus: 'failed' as const,
        updatedAt: new Date()
      } : null);
    }
  };

  const handleStopCheckin = async () => {
    if (!checkin) return;

    try {
      console.log(`⏹️ Check-in durduruluyor: ${checkin.pnr}`);

      const jobId = checkin.ticketRequestId || checkin.id;

      const response = await fetch(`${process.env.NEXT_PUBLIC_API_URL || 'http://localhost:3002'}/api/checkin/${jobId}`, {
        method: 'DELETE'
      });

      if (response.ok) {
        const result = await response.json();
        console.log('Check-in durdurma sonucu:', result);

        // Durumu waiting'e geri çevir
        setCheckin(prev => prev ? {
          ...prev,
          checkinStatus: 'waiting' as const,
          updatedAt: new Date()
        } : null);

        console.log(`⏹️ Check-in durduruldu: ${checkin.pnr}`);
      } else {
        const errorText = await response.text();
        console.error('Durdurma hatası:', errorText);
        throw new Error(`Check-in durdurulamadı: ${response.status}`);
      }

    } catch (error) {
      console.error(`❌ Check-in durdurma hatası: ${checkin.pnr}`, error);
    }
  };

  const handleDownloadAllBoardingPasses = () => {
    checkin?.passengerResults.forEach(result => {
      if (result.boardingPassUrl) {
        window.open(result.boardingPassUrl, '_blank');
      }
    });
  };

  const handleViewScreenshot = (screenshot: string) => {
    window.open(screenshot, '_blank');
  };

  const getProgressIcon = (status: CheckinProgress['status'], theme: { text: string; border: string }) => {
    switch (status) {
      case 'completed': return <CheckCircle className="h-5 w-5 text-green-600" />;
      case 'failed': return <XCircle className="h-5 w-5 text-red-600" />;
      case 'processing': return <Loader className={`h-5 w-5 ${theme.text} animate-spin`} />;
      default: return <div className={`h-5 w-5 rounded-full border-2 ${theme.border}`} />;
    }
  };

  const getLogIcon = (level: CheckinLog['level']) => {
    switch (level) {
      case 'error': return <XCircle className="h-4 w-4 text-red-500" />;
      case 'warn': return <AlertTriangle className="h-4 w-4 text-yellow-500" />;
      case 'info': return <Activity className="h-4 w-4 text-blue-500" />;
      default: return <Activity className="h-4 w-4 text-gray-500" />;
    }
  };

  const statusOptions = [
    { value: 'waiting', label: 'Bekliyor' },
    { value: 'ready', label: 'Hazır' },
    { value: 'processing', label: 'İşleniyor' },
    { value: 'completed', label: 'Tamamlandı' },
    { value: 'failed', label: 'Başarısız' },
    { value: 'expired', label: 'Süresi Geçti' },
    { value: 'cancelled', label: 'İptal' }
  ];

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="animate-pulse space-y-4">
          <div className="h-8 bg-gray-200 rounded w-64"></div>
          <div className="h-64 bg-gray-200 rounded w-96"></div>
        </div>
      </div>
    );
  }

  if (!checkin) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <XCircle className="h-16 w-16 text-gray-400 mx-auto mb-4" />
          <h2 className="text-xl font-semibold text-gray-900 mb-2">Check-in Kaydı Bulunamadı</h2>
          <p className="text-gray-600 mb-4">
            Bu ID ile ilişkili check-in kaydı bulunamadı.
          </p>
          <button
            onClick={() => window.location.href = '/checkin'}
            className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
          >
            Check-in Listesine Dön
          </button>
        </div>
      </div>
    );
  }

  const theme = getAirlineTheme(checkin?.airline || 'THY');

  return (
    <DetailLayout
      title={`Check-in #${checkin.id}`}
      subtitle={
        <div className="flex items-center space-x-2">
          <span className="text-2xl">{theme.logo}</span>
          <span className={`font-medium ${theme.text}`}>{theme.name}</span>
          <span className="text-gray-500">•</span>
          <span>{checkin.flightNumber}</span>
          <span className="text-gray-500">•</span>
          <span>PNR: {checkin.pnr}</span>
        </div>
      }
      backUrl="/checkin"
      recordId={checkin.id}
      recordType="ticket"
      comments={comments}
      onAddComment={handleAddComment}
      onStatusChange={handleStatusChange}
      statusOptions={statusOptions}
      currentStatus={checkin.checkinStatus}
      currentUser={{ id: 'agent1', name: 'Acente', type: 'agent' }}
      customer={{
        id: checkin.customer?.id,
        name: checkin.customer?.type === 'individual'
          ? `${checkin.customer?.firstName || ''} ${checkin.customer?.lastName || ''}`
          : checkin.customer?.companyName || 'Kurumsal Müşteri',
        type: checkin.customer?.type
      }}
    >
      <div className="space-y-6">
        {/* Progress Tracker */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <div className="flex items-center justify-between mb-6">
            <div className="flex items-center space-x-4">
              <h3 className="text-lg font-semibold text-gray-900">Check-in Süreci</h3>
              <div className="flex items-center space-x-1">
                {connectionStatus === 'connected' ? (
                  <><Wifi className="h-4 w-4 text-green-500" /><span className="text-xs text-green-600">Canlı</span></>
                ) : connectionStatus === 'connecting' ? (
                  <><Loader className="h-4 w-4 text-yellow-500 animate-spin" /><span className="text-xs text-yellow-600">Bağlanıyor</span></>
                ) : (
                  <><WifiOff className="h-4 w-4 text-red-500" /><span className="text-xs text-red-600">Bağlantı Yok</span></>
                )}
              </div>
            </div>
            <div className="flex items-center space-x-2">
              {checkin.checkinStatus === 'processing' && (
                <span className="flex items-center text-sm text-blue-600">
                  <Loader className="h-4 w-4 mr-1 animate-spin" />
                  İşlem devam ediyor...
                </span>
              )}
              <button
                onClick={() => setShowLogs(!showLogs)}
                className="flex items-center px-3 py-1 text-sm bg-gray-100 text-gray-700 rounded-md hover:bg-gray-200"
              >
                <Eye className="h-4 w-4 mr-1" />
                {showLogs ? 'Logları Gizle' : 'Logları Göster'}
              </button>
            </div>
          </div>

          <div className="space-y-4">
            {progress.map((step, index) => {
              // Bu adıma ait logları filtrele
              const stepLogs = logs.filter(log => {
                const percentage = parseInt(log.action);
                if (step.step === 1) return percentage >= 65 && percentage < 70;
                if (step.step === 2) return percentage >= 70 && percentage < 75;
                if (step.step === 3) return percentage >= 75 && percentage < 85;
                if (step.step === 4) return percentage >= 85 && percentage < 95;
                if (step.step === 5) return percentage >= 95;
                return false;
              });

              return (
                <div key={step.step} className="flex items-start space-x-4">
                  <div className="flex-shrink-0 mt-1">
                    {getProgressIcon(step.status, theme)}
                  </div>

                  <div className="flex-1 min-w-0">
                    <div className="flex items-center justify-between">
                      <h4 className="text-sm font-medium text-gray-900">
                        {step.step}. {step.title}
                      </h4>
                      {step.timestamp && (
                        <span className="text-xs text-gray-500">
                          {step.timestamp.toLocaleTimeString('tr-TR')}
                        </span>
                      )}
                    </div>

                    <p className="text-sm text-gray-600 mt-1">{step.description}</p>

                    {step.details && (
                      <p className="text-xs text-gray-500 mt-1">{step.details}</p>
                    )}

                    {/* Bu adıma ait detaylı loglar */}
                    {stepLogs.length > 0 && (
                      <div className="mt-2 ml-4 space-y-1 border-l-2 border-gray-200 pl-3">
                        {stepLogs.map((log) => (
                          <div key={log.id} className="flex items-start space-x-2 text-xs">
                            <span className="text-gray-400">{log.timestamp.toLocaleTimeString('tr-TR')}</span>
                            <span className={`flex-1 ${log.level === 'error' ? 'text-red-600' : 'text-gray-600'}`}>
                              {log.message}
                            </span>
                          </div>
                        ))}
                      </div>
                    )}

                    {step.error && (
                      <div className="mt-2 p-2 bg-red-50 border border-red-200 rounded">
                        <p className="text-sm text-red-800">{step.error}</p>
                      </div>
                    )}

                    {step.screenshot && (
                      <button
                        onClick={() => handleViewScreenshot(step.screenshot!)}
                        className="mt-1 text-xs text-blue-600 hover:text-blue-800 underline"
                      >
                        Ekran görüntüsünü göster
                      </button>
                    )}
                  </div>

                  {index < progress.length - 1 && (
                    <div className="absolute left-7 mt-8 w-px h-6 bg-gray-200" />
                  )}
                </div>
              );
            })}
          </div>

        </div>
        {/* Status Card */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <div className="flex items-center justify-between mb-4">
            <h3 className="text-lg font-semibold text-gray-900">Check-in Durumu</h3>
            <div className="flex items-center space-x-3">
              {renderStatusBadge(checkin.checkinStatus, checkinStatusConfig)}
              {checkin.isAutoCheckinEnabled && (
                <span className={`px-2 py-1 text-xs ${theme.secondary} ${theme.text} rounded-full`}>
                  Otomatik
                </span>
              )}
            </div>
          </div>
          
          <div className="grid grid-cols-3 gap-4 text-sm">
            <div>
              <span className="text-gray-500">Check-in Açılış:</span>
              <p className="font-medium">
                {checkin.checkinOpenTime?.toLocaleString('tr-TR')}
              </p>
            </div>
            <div>
              <span className="text-gray-500">Tamamlanma:</span>
              <p className="font-medium">
                {checkin.checkinCompletedAt ? 
                  checkin.checkinCompletedAt.toLocaleString('tr-TR') : 
                  'Henüz tamamlanmadı'
                }
              </p>
            </div>
            <div>
              <span className="text-gray-500">Deneme Sayısı:</span>
              <p className="font-medium">{checkin.checkinAttempts}</p>
            </div>
          </div>

          {checkin.errorMessage && (
            <div className="mt-4 p-3 bg-red-50 border border-red-200 rounded-lg">
              <div className="flex items-center space-x-2">
                <AlertTriangle className="h-4 w-4 text-red-600" />
                <span className="text-sm font-medium text-red-900">Hata:</span>
              </div>
              <p className="text-sm text-red-800 mt-1">{checkin.errorMessage}</p>
            </div>
          )}
        </div>

        {/* Customer Info */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
            <User className="h-5 w-5 mr-2" />
            Müşteri Bilgileri
          </h3>
          {renderCustomer(checkin.customer)}
        </div>

        {/* Flight Info */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
            <Plane className="h-5 w-5 mr-2" />
            Uçuş Bilgileri
          </h3>
          <div className="grid grid-cols-2 gap-6">
            <div>
              <div className="flex items-center space-x-2 mb-2">
                <MapPin className="h-4 w-4 text-gray-400" />
                <span className="text-sm text-gray-500">Kalkış</span>
              </div>
              <p className="text-lg font-semibold">{checkin.departureAirport}</p>
            </div>
            <div>
              <div className="flex items-center space-x-2 mb-2">
                <MapPin className="h-4 w-4 text-gray-400" />
                <span className="text-sm text-gray-500">Varış</span>
              </div>
              <p className="text-lg font-semibold">{checkin.arrivalAirport}</p>
            </div>
          </div>
          
          <div className="mt-6 grid grid-cols-3 gap-4">
            <div>
              <div className="flex items-center space-x-2 mb-2">
                <Calendar className="h-4 w-4 text-gray-400" />
                <span className="text-sm text-gray-500">Tarih</span>
              </div>
              <p className="font-medium">{checkin.departureDate.toLocaleDateString('tr-TR')}</p>
            </div>
            <div>
              <div className="flex items-center space-x-2 mb-2">
                <Clock className="h-4 w-4 text-gray-400" />
                <span className="text-sm text-gray-500">Saat</span>
              </div>
              <p className="font-medium">{checkin.departureTime}</p>
            </div>
            <div>
              <div className="flex items-center space-x-2 mb-2">
                <FileText className="h-4 w-4 text-gray-400" />
                <span className="text-sm text-gray-500">PNR</span>
              </div>
              <p className="font-mono font-medium text-blue-600">{checkin.pnr}</p>
            </div>
          </div>
        </div>

        {/* Passengers & Check-in Results */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4">Yolcu Check-in Durumu</h3>
          <div className="space-y-4">
            {checkin.passengers?.map((passenger, index) => {
              const result = checkin.passengerResults?.find(r => r.passengerId === passenger.id);
              return (
                <div key={passenger.id} className="border border-gray-200 rounded-lg p-4">
                  <div className="flex items-center justify-between mb-3">
                    <div>
                      <p className="font-medium text-gray-900">
                        {passenger.firstName} {passenger.lastName}
                        <span className={`ml-2 px-2 py-1 text-xs ${theme.secondary} ${theme.text} rounded`}>
                          Yolcu {index + 1}
                        </span>
                      </p>
                      <p className="text-sm text-gray-500">{passenger.email}</p>
                    </div>
                    {result && renderStatusBadge(result.checkinStatus, checkinStatusConfig)}
                  </div>
                  
                  {result && (
                    <div className="grid grid-cols-2 gap-4 mt-3">
                      {result.seat && (
                        <div className="flex items-center space-x-2">
                          <Armchair className="h-4 w-4 text-green-600" />
                          <span className="text-sm text-gray-600">Koltuk:</span>
                          <span className="font-medium text-green-600">{result.seat}</span>
                        </div>
                      )}
                      
                      {result.boardingPassUrl && (
                        <div className="flex items-center space-x-2">
                          <Download className="h-4 w-4 text-blue-600" />
                          <button
                            onClick={() => handleDownloadBoardingPass(passenger.id)}
                            className="text-sm text-blue-600 hover:text-blue-800 underline"
                          >
                            Biniş Kartını İndir
                          </button>
                        </div>
                      )}
                    </div>
                  )}
                  
                  {result?.errorMessage && (
                    <div className="mt-3 p-2 bg-red-50 border border-red-200 rounded">
                      <p className="text-sm text-red-800">{result.errorMessage}</p>
                    </div>
                  )}
                </div>
              );
            })}
          </div>
        </div>

        {/* SMS Status */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
            <Smartphone className="h-5 w-5 mr-2" />
            SMS Bildirimi
          </h3>
          <div className="flex items-center justify-between">
            <span className="text-gray-600">SMS Durumu:</span>
            {renderStatusBadge(checkin.smsStatus, {
              sent: { color: 'bg-green-100 text-green-800', label: 'Gönderildi' },
              failed: { color: 'bg-red-100 text-red-800', label: 'Başarısız' },
              pending: { color: 'bg-gray-100 text-gray-800', label: 'Bekliyor' }
            })}
          </div>
        </div>

        {/* Actions */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4">İşlemler</h3>
          <div className="flex flex-wrap gap-3">
            {/* Check-in Başlat */}
            {['waiting', 'ready', 'failed'].includes(checkin.checkinStatus) && (
              <button
                onClick={handleStartCheckin}
                className="flex items-center px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700"
              >
                <Play className="h-4 w-4 mr-2" />
                Check-in Başlat
              </button>
            )}

            {/* Check-in Durdur */}
            {checkin.checkinStatus === 'processing' && (
              <button
                onClick={handleStopCheckin}
                className="flex items-center px-4 py-2 bg-orange-600 text-white rounded-lg hover:bg-orange-700"
              >
                <Square className="h-4 w-4 mr-2" />
                Durdur
              </button>
            )}

            {/* Yeniden Dene */}
            {checkin.checkinStatus === 'failed' && (
              <button
                onClick={handleRetryCheckin}
                disabled={isRetrying}
                className={`flex items-center px-4 py-2 ${theme.primary} text-white rounded-lg hover:opacity-90 disabled:opacity-50 disabled:cursor-not-allowed`}
              >
                {isRetrying ? (
                  <Loader className="h-4 w-4 mr-2 animate-spin" />
                ) : (
                  <RotateCcw className="h-4 w-4 mr-2" />
                )}
                {isRetrying ? 'Yeniden Deneniyor...' : 'Tekrar Dene'}
              </button>
            )}

            {/* Biniş Kartları İndirme */}
            {checkin.passengerResults?.some(r => r.boardingPassUrl) && (
              <button
                onClick={handleDownloadAllBoardingPasses}
                className="flex items-center px-4 py-2 bg-indigo-600 text-white rounded-lg hover:bg-indigo-700"
              >
                <Download className="h-4 w-4 mr-2" />
                Tüm Biniş Kartlarını İndir
              </button>
            )}

            {/* SMS Gönder */}
            <button
              onClick={() => {
                // SMS gönderme işlemi
                console.log('SMS gönderiliyor...');
              }}
              className="flex items-center px-4 py-2 bg-purple-600 text-white rounded-lg hover:bg-purple-700"
            >
              <Smartphone className="h-4 w-4 mr-2" />
              SMS Gönder
            </button>

          </div>
        </div>

        {/* Notes */}
        {checkin.notes && (
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
            <h3 className="text-lg font-semibold text-gray-900 mb-4">Notlar</h3>
            <p className="text-gray-700">{checkin.notes}</p>
          </div>
        )}
      </div>
    </DetailLayout>
  );
}